"""Parser for command line option"""
import getopt
import json
import os
import shutil
import sys

from loguru import logger

from config.app_config import AppConfig
from features.min_max_stats_finder import min_max_scores, write_vars_to_file

def show_chapters(chapters):
    """Print a summary of the content of the json file"""
    print("=============== CHAPITRES ===============-")
    for chapter in chapters:
        print(f"Id: \t\t{chapter['Id']}")
        print(f"Nom: \t\t{chapter['Name']}")
        print(f"Scènes({len(chapter['Scenes'])})")
        for scenes in chapter['Scenes']:
            print(f"\t\t\t{scenes['Title']} ({len(scenes['Interactions'])})")
        print("======================================")


def normalize_string(text):
    """Normalize an input string"""
    return text.lower().replace(" ", "_").replace("é", "e") \
        .replace("ê", "e").replace("è", "e").replace("à", "a")
        
def normlize_dialog(text):
    return text.replace("\n","")


def main(argv):
    """Main function"""

    app_config = AppConfig()

    inputfile = app_config.get("INPUT_FILE_NAME")
    output_folder = app_config.get("OUTPUT_FOLDER_NAME")
    tree = False
    scores = False

    try:
        opts, _ = getopt.getopt(argv, "hsti:o:", ["ifile=", "ofile=", "tree","scores"])
    except getopt.GetoptError:
        print('EODLJsonParser.py -i <inputfile> -o <outputFolder>')
        sys.exit(2)
    for opt, arg in opts:
        if opt == '-h':
            print('EODLJsonParser.py -i <inputfile> -o <outputFolder>')
            sys.exit()
        elif opt in ("-i", "--ifile"):
            inputfile = arg
        elif opt in ("-o", "--ofile"):
            output_folder = arg
        elif opt in ("-t", "--tree"):
            tree = True
        elif opt in ("-s", "--scores"):
            scores = True

    with open(inputfile, 'r', encoding="utf-8") as j:
        contents = json.loads(j.read())

        # If --tree argument exists
        if tree:
            show_chapters(contents["Chapters"])
            sys.exit(0)
            
        #if --scores argument exists
        if scores:
            scores = min_max_scores(contents)
            write_vars_to_file(scores)
            sys.exit(0)

        # if output folder already exist, we delete it
        if os.path.exists(output_folder):
            shutil.rmtree(output_folder)

        logger.debug(f"Creating {output_folder} folder...")
        os.mkdir(output_folder)

        # Iterate over chapters
        for idx_chapter, chapter in enumerate(contents["Chapters"]):
            chapter_name = normalize_string(chapter["Name"])
            chapter_folder = os.path.join(output_folder, chapter_name)
            os.mkdir(chapter_folder)

            # create intro file
            intro_file_name = chapter_name + "_intro.nani"
            intro_file_path = os.path.join(chapter_folder, intro_file_name)
            with open(intro_file_path, 'w', encoding='utf-8') as f:
                f.write("@hideUI LoadingUI\n")
                f.write("@back Hospital\n")
                f.write("@spawn Blur wait:false\n")
                f.write("@printer Fullscreen\n")
                f.write(f"{chapter['Name']}\n")
                f.write(f"{chapter['Intro']}\n")
                f.write(
                    f"@goto {normalize_string(chapter['Scenes'][0]['Title'])}")

            for idx_scene, scene in enumerate(chapter["Scenes"]):
                scene_name = normalize_string(scene["Title"])
                scene_path = os.path.join(chapter_folder, scene_name)
                os.mkdir(scene_path)

                # We create and write the scene in a naninovel file
                naninovel_file = scene_name + ".nani"
                naninovel_file_path = os.path.join(scene_path, naninovel_file)

                with open(naninovel_file_path, 'w', encoding='utf-8') as f:

                    # First we create environment and introduction
                    f.write("@hideChars remove:true\n")
                    f.write("@hideUI LoadingUI\n")
                    f.write("@back Hospital\n")
                    f.write("@spawn Blur wait:false\n")
                    f.write("@printer Fullscreen\n")
                    
                    # Reseting player variables
                    f.write("@set Empathy=0\n")
                    f.write("@set Respect=0\n")
                    f.write("@set Compassion=0\n")
                    f.write("@set Hope=0\n")
                    f.write("@set Authenticity=0\n")
                    f.write(f"@set current_scene=\"{scene_name}\"\n")
                    
                    f.write(f"{scene['SceneIntroduction']}\n\n")

                    f.write("@resetText\n")
                    f.write("@printer Dialogue\n")
                    # Iterate over interactions
                    for interaction in scene["Interactions"]:
                        f.write(f"# {interaction['Id']}\n")
                        if interaction['Actor']['Name'] != "Claude": # We display the character ONLY if it's not the main character
                            f.write(f"@char {interaction['Actor']['Name']}.{interaction['Actor']['Emotion']}\n")
                        f.write(
                            f"{interaction['Actor']['Name']}: {interaction['Text']}\n\n")
                        
                        # Iterate over choices 
                        for choice in interaction['Responses']:

                            stat_update = f"set:Authenticity+={choice['Authenticity']};Respect+={choice['Respect']};Compassion+={choice['Compassion']};Hope+={choice['Hope']};Empathy+={choice['Empathy']}"
                            choice_text = normlize_dialog(choice['Text'])
                            #Check if this is an end choice of the current script
                            if choice['NextInteractionID'] == int(app_config.get("LAST_INTERACTION")):
                                #Check if the current scene is the last one from the current chapter
                                if idx_scene == len(chapter["Scenes"]) - 1:
                                    # Check if the current chatper is the last one from the scenario
                                    if idx_chapter == len(contents["Chapters"]) -1:
                                        #f.write(
                                        #f"@choice \"{choice['Text']}\" {stat_update} goto:FIN\n")
                                        f.write(
                                        f"@choice \"{choice_text}\" {stat_update};next_scene=\"FIN\" goto:Scores \n")
                                    else:
                                        #f.write(
                                        #f"@choice \"{choice['Text']}\" {stat_update} goto:{normalize_string(contents['Chapters'][idx_chapter + 1]['Name']) + '_intro'}\n")
                                        f.write(
                                        f"@choice \"{choice_text}\" {stat_update};next_scene=\"{normalize_string(contents['Chapters'][idx_chapter + 1]['Name']) + '_intro'}\" goto:Scores \n")
                                else:
                                    #f.write(
                                    #    f"@choice \"{choice['Text']}\" {stat_update} goto:{normalize_string(chapter['Scenes'][idx_scene+1]['Title'])}\n")
                                    f.write(
                                        f"@choice \"{choice_text}\" {stat_update};next_scene=\"{normalize_string(chapter['Scenes'][idx_scene+1]['Title'])}\" goto:Scores \n")
                            else:
                                f.write(
                                    f"@choice \"{choice_text}\" {stat_update} goto:.{choice['NextInteractionID']}\n")

                        f.write("@stop\n\n")
        logger.debug("DONE !")


if __name__ == "__main__":
    main(sys.argv[1:])
