#pragma once

#include <stdexcept>
#include <vector>
#include <cstdint>
#include <string>

#ifdef _WIN32
#include <Windows.h>
#endif

#define MCP2221_NAME            "MCP2221 USB-I2C/UART Combo"
#define I2C_SPEED_100KHZ        100000
#define I2C_SPEED_400KHZ        400000
#define I2C_TIMEOUT_1MS         1

namespace MCP2221A {

    class MCP2221AException : public std::exception {
        private:

        std::string msg_;

        public:

        MCP2221AException(std::string msg) : msg_(msg) {}

        virtual const char* what() const throw()
        {
            return msg_.c_str();
        }
    };

    class I2CMasterException : public MCP2221AException { using MCP2221AException::MCP2221AException; };
    class SerialException : public MCP2221AException { using MCP2221AException::MCP2221AException; };
    class TimeoutException : public MCP2221AException { using MCP2221AException::MCP2221AException; };

    /**
     * @brief I2CMaster class to control the MCP2221A I2C port
     * 
     */
    class I2CMaster {

        private:

#ifdef __linux__
        int fd_;
#elif _WIN32
        void* handle_;
#endif
        std::string mcp2221a_path_;
        uint8_t current_slave_addr_;
        std::vector<int> slaves_addr_;
        unsigned int i2c_speed_;
        unsigned char i2c_timeout_ms_;

        /**
         * @brief Scan the I2C bus looking for available devices
         * 
         */
        void scan_bus();

        /**
         * @brief Set the slave address
         * 
         * @param addr 7 bit slave address
         * 
         * @throw I2CMasterException if it fails
         */
        void set_slave_address(uint8_t addr);

        /**
         * @brief Not implemented
         * 
         * @param addr 
         * @return true 
         * @return false 
         */
        bool check_slave_connected(uint8_t addr);

        public:

        /**
         * @brief Construct a new I2CMaster object
         * 
         * @param port I2C device path. In linux it should be /dev/i2c-X where X is a number 0,1,2,etc...
         * 
         * @throw I2CMasterException if something fails.
         */
#ifdef _WIN32
        I2CMaster(unsigned int speed = I2C_SPEED_100KHZ, unsigned char timeout_ms = I2C_TIMEOUT_1MS);
#elif __linux__
        I2CMaster(unsigned int speed = I2C_SPEED_100KHZ, unsigned char timeout_ms = I2C_TIMEOUT_1MS);
#endif
        /**
         * @brief Destroy the I2CMaster object
         * 
         */
        ~I2CMaster();

        /**
         * @brief Write data to the I2C slave.
         * 
         * @param slave_addr Slave 7-bit address
         * @param data Byte array of data to send
         * @param length Number of bytes to send
         * 
         * @throw I2CMasterException if an error occurs
         */
        void send(uint8_t slave_addr, uint8_t *data, size_t length);

        /**
         * @brief Read data from the I2C slave.
         * 
         * @param slave_addr Slave 7-bit address
         * @param data Buffer to be filled
         * @param length Number of bytes to read
         * 
         * @throw I2CMasterException if an error occurs
         */
        void receive(uint8_t slave_addr, uint8_t *data, size_t length);

        /**
         * @brief Read data from a specific register of the I2C slave
         * 
         * @param slave_addr 
         * @param reg_addr 
         * @param data 
         * @param length 
         */
        void read_register(uint8_t slave_addr, uint8_t reg_addr, uint8_t *data, size_t length);

        /**
         * @brief Get the slaves addresses found on the I2C bus
         * 
         * @param force_scan Re-scan the bus
         * 
         * @return std::vector<int> Slave addresses. List can be empty if no slave are found
         */
        std::vector<int> get_slaves_addresses(bool force_scan = false);

        /**
         * @brief Set the I2C bus speed
         * 
         * @param speed Speed in Hz. Supported speeds are 100000 (standard mode) and 400000 (fast mode)
         * 
         * @throw I2CMasterException if an error occurs
         */
        void set_speed(uint32_t speed);
    };

    /**
     * @brief Serial class to control the MCP2221A UART functionalities
     * 
     */
    class Serial {

        public:

        enum BAUD {
            B_300,
            B_1200,
            B_2400,
            B_4800,
            B_9600,
            B_19200,
            B_38400,
            B_57600,
            B_115200,
            B_230400,
            B_460800
        };
        
        private:
#ifdef __linux__
        int fd_;
        std::string port_;
        uint8_t read_timeout_;
#elif _WIN32
        HANDLE handle_;
        std::wstring port_;
        DWORD read_timeout_;
#endif
        BAUD baudrate_;

        public:
        /**
         * @brief Construct a new Serial. This object gives access to the MCP2221A serial port
         * 
         * @param port Port name. In linux it should be ttyACM<X> where X is a number 0,1,2,etc...
         * @param baudrate Baudrate. Supported baudrates are listed in the BAUD enum
         * 
         * @throw SerialException if something fails.
         */
#ifdef _WIN32
        Serial(std::wstring port = std::wstring(L"COM3"), BAUD baudrate = B_115200);
#elif __linux__
        Serial(std::string port = "ttyACM0", BAUD baudrate = B_115200);
#endif
        /**
         * @brief Destroy the Serial.
         * 
         */
        ~Serial();
        
        /**
         * @brief Write data to the serial port
         * 
         * @param data Byte array of data to send
         * @param length Number of bytes to send
         * 
         * @throw SerialException if an error occurs
         */
        void send(uint8_t *data, size_t length);

        /**
         * @brief Read data from the serial port
         * 
         * @param data Buffer to be filled.
         * @param length Number of bytes to read
         * 
         * @throw SerialException if an error occurs
         * @throw TimeoutException if no data or not enough data is received
         */
        void receive(uint8_t *data, size_t length);

        /**
         * @brief Set the baudrate of the serial port
         * 
         * @param baudrate Serial port baudrate
         * 
         * @throw SerialException if an error occurs
         */
        void set_baudrate(BAUD baudrate);

#ifdef __linux__
        /**
         * @brief Set the timeout 
         * 
         * @param timeout Timeout value in deciseconds (0.1s). A value of 0 will be interpreted as no timeout. Max value is 255 or 25.5s
         * 
         * @throw SerialException if an error occurs
         */
        void set_timeout(uint8_t timeout);
#elif _WIN32
        /**
         * @brief Set the timeout 
         * 
         * @param timeout Timeout value in milliseconds.
         * 
         * @throw SerialException if an error occurs
         */
        void set_timeout(DWORD timeout_ms);
#endif
    };
}