#include <libusb.h>

#include <cstdio>
#include <cstring>

#include "mcp2221a_hid.h"

#define MCP2221A_VID                0x04D8
#define MCP2221A_PID                0x00DD
#define MCP2221A_HID_IFACE          2
#define MCP2221A_HID_ENDPOINT_IN    0x83
#define MCP2221A_HID_ENDPOINT_OUT   0x03

#define HID_PACKET_SIZE             64
#define HID_STATUS_SET_PARAMS       0x10
#define HID_CANCEL_CURRENT_TRANSFER 0x10
#define HID_SET_I2C_SPEED           0x20

namespace MCP2221A {

    static std::string error_string(int code)
    {
        switch (code) {
            case LIBUSB_SUCCESS:
                return "Success";
            case LIBUSB_ERROR_IO:
                return "Input/output error";
            case LIBUSB_ERROR_INVALID_PARAM:
                return "Invalid parameter";
            case LIBUSB_ERROR_ACCESS:
                return "Access denied (insufficient permissions)";
            case LIBUSB_ERROR_NO_DEVICE:
                return "No such device (it may have been disconnected)";
            case LIBUSB_ERROR_NOT_FOUND:
                return "Device not found";
            case LIBUSB_ERROR_BUSY:
                return "Resource busy";
            case LIBUSB_ERROR_TIMEOUT:
                return "Operation timed out";
            case LIBUSB_ERROR_OVERFLOW:
                return "Overflow";
            case LIBUSB_ERROR_PIPE:
                return "Pipe error";
            case LIBUSB_ERROR_INTERRUPTED:
                return "System call interrupted (perhaps due to signal)";
            case LIBUSB_ERROR_NO_MEM:
                return "Insufficient memory";
            case LIBUSB_ERROR_NOT_SUPPORTED:
                return "Operation not supported or unimplemented on this platform";
            case LIBUSB_ERROR_OTHER:
                return "Other error";
            default:
                return "Unknown error";
        }
    }

    static int hid_open_device(libusb_device_handle*& handle, libusb_context*& ctx)
    {
        ctx = NULL;
        handle = NULL;
        int ret = 0;

        if ((ret = libusb_init(&ctx)) < 0) {
            goto out;
        }

        handle = libusb_open_device_with_vid_pid(ctx, MCP2221A_VID, MCP2221A_PID);
        if (!handle) {
            ret = LIBUSB_ERROR_NOT_FOUND;
            goto out_open_device;
        }
        
        if (libusb_kernel_driver_active(handle, MCP2221A_HID_IFACE) == 1) {
            if ((ret = libusb_detach_kernel_driver(handle, MCP2221A_HID_IFACE)) < 0) {
                goto out_detach_kernel;
            }
        }

        if ((ret = libusb_claim_interface(handle, MCP2221A_HID_IFACE)) < 0) {
            goto out_release_interface;
        }

        goto out;

out_release_interface:
out_detach_kernel:
        libusb_close(handle);
        handle = NULL;
out_open_device:
        libusb_exit(ctx);
        ctx = NULL;
out:
        return ret;
    }

    static int hid_close_device(libusb_device_handle* handle, libusb_context* ctx)
    {
        int ret = 0;

        if (handle) {
            if((ret = libusb_release_interface(handle, MCP2221A_HID_IFACE)) < 0) {
                return ret;
            }

            if((ret = libusb_attach_kernel_driver(handle, MCP2221A_HID_IFACE)) < 0) {
                return ret;
            }

            libusb_close(handle);
        }

        if (ctx) {
            libusb_exit(ctx);
        }

        return 0;
    }

    static int hid_read_data(libusb_device_handle* handle, uint8_t* data, int length)
    {
        int ret = 0, transferred = 0;

        ret = libusb_interrupt_transfer(handle,
                                      MCP2221A_HID_ENDPOINT_IN, // endpoint
                                      data,
                                      length,
                                      &transferred,
                                      1000);

        if (ret != LIBUSB_SUCCESS) {
            return ret;
        }

        return transferred;
    }

    static int hid_write_data(libusb_device_handle* handle, uint8_t* data, int length)
    {
        int ret = 0, transferred = 0;

        ret = libusb_interrupt_transfer(handle,
                                      MCP2221A_HID_ENDPOINT_OUT, // endpoint
                                      data,
                                      length,
                                      &transferred,
                                      1000);

        if (ret != LIBUSB_SUCCESS) {
            return ret;
        }

        return transferred;
    }

    void hid_set_i2c_speed(unsigned int speed)
    {
        libusb_context *ctx = NULL;
        libusb_device_handle *handle = NULL;
        uint8_t data[HID_PACKET_SIZE] = {0};
        status stat = {0};
        std::string err;
        int ret = 0, transferred = 0;

        if ((ret = hid_open_device(handle, ctx)) < 0) {
            err = "Failed to open device: " + error_string(ret);
            goto error;
        }

        data[0] = HID_STATUS_SET_PARAMS;
        data[3] = HID_SET_I2C_SPEED;
        data[4] = (uint8_t)((12000000 / speed) - 2);

        if ((ret = hid_write_data(handle, data, sizeof(data))) < 0) {
            err = "Failed to send set speed request: " + error_string(ret);
            goto error;
        }
        if (ret != sizeof(data)) {
            err = "Failed to send complete set speed request";
            goto error;
        }

        if ((ret = hid_read_data(handle, stat.raw, sizeof(stat.raw))) < 0) {
            err = "Failed to get set speed response: " + error_string(ret);
            goto error;
        }
        if (ret != sizeof(stat.raw)) {
            err = "Failed to read complete set speed response";
            goto error;
        }

        if (stat.success == 0 && stat.i2c_speed_set == HID_SET_I2C_SPEED) {
            hid_close_device(handle, ctx);
        } else {
            err = "Device reported failure to set I2C speed";
        }

        return;
error:
        hid_close_device(handle, ctx);
        throw HIDException(err);
    }

    void hid_get_status(status& stat)
    {
        int ret = 0;
        libusb_context *ctx = NULL;
        libusb_device_handle *handle = NULL;
        uint8_t data[HID_PACKET_SIZE] = {0};
        int transferred = 0;
        std::string err;

        if ((ret = hid_open_device(handle, ctx)) < 0) {
            err = "Failed to open device: " + error_string(ret);
            goto error;
        }

        data[0] = HID_STATUS_SET_PARAMS;

        if ((ret = hid_write_data(handle, data, sizeof(data))) < 0) {
            err = "Failed to send status request: " + error_string(ret);
            goto error;
        }
        if (ret != sizeof(data)) {
            err = "Failed to send complete status request";
            goto error;
        }

        if ((ret = hid_read_data(handle, stat.raw, sizeof(stat.raw))) < 0) {
            err = "Failed to get status response: " + error_string(ret);
            goto error;
        }
        if (ret != sizeof(stat.raw)) {
            err = "Failed to read complete status response";
            goto error;
        }

        hid_close_device(handle, ctx);

        return;
error:
        hid_close_device(handle, ctx);
        throw HIDException(err);
    }
}