#include <iostream>

extern "C" {
    #include <dirent.h>
    #include <unistd.h>
    #include <sys/stat.h>
    #include <fcntl.h>
    #include <errno.h>
    #include <string.h>
    #include <stdlib.h>
    #include <linux/i2c-dev.h>
    #include <i2c/smbus.h>
    #include <sys/ioctl.h>
    #include <libusb.h>
}

#include "mcp2221a.h"
#include "mcp2221a_hid.h"

#define MCP2221_I2C_NAME    "MCP2221 usb-i2c bridge"
#define SYSFS_I2C_PATH      "/sys/class/i2c-dev/"
#define I2C_MAX_ADDR        0x80

using namespace std;

namespace MCP2221A {

    vector<string> find_i2c_devices()
    {
        DIR *target = NULL;
        struct dirent *about;
        struct stat file;
        vector<string> i2c_devs;

        target = opendir(SYSFS_I2C_PATH);
        if (!target) {
            goto exit;
        }

        while (about = readdir(target)) {
            size_t found = string(about->d_name).find("i2c");
            if (string::npos != found)
                i2c_devs.push_back(about->d_name);
        }

        closedir(target);

    exit:
        return i2c_devs;
    }

    string get_i2c_device_name(string i2c_dev_nr)
    {
        int fd = 0, ret = 0;
        char buf;
        string device_name = "";
        string device_name_path = SYSFS_I2C_PATH + i2c_dev_nr + "/name";
        
        if ((fd = open(device_name_path.c_str(), O_RDONLY)) < 0) {
            cout << "Failed to open " << device_name_path << ": " << strerror(errno) << endl;
            return "";
        }

        while ((ret = read(fd, &buf, sizeof(buf)))) {
            device_name += buf;
        }
        
        if (ret < 0) {
            cout << "Failed to read " << device_name_path << ": " << strerror(errno) << endl;
            return "";
        }

        close(fd);

        return device_name;
    }

    I2CMaster::I2CMaster(unsigned int speed, unsigned char timeout_ms) : fd_(0), mcp2221a_path_(""), current_slave_addr_(0)
    {
        vector<string> i2c_devices = find_i2c_devices();

        for (auto& dev: i2c_devices) {
            if (get_i2c_device_name(dev).find(MCP2221_I2C_NAME) != string::npos) {
                mcp2221a_path_ = "/dev/" + dev;
                break;
            }
        }

        try {
            hid_set_i2c_speed(speed);
        } catch (const HIDException& ex) {
            throw I2CMasterException(ex.what());
        }

        if (!mcp2221a_path_.empty()) {
            if ((fd_ = open(mcp2221a_path_.c_str(), O_RDWR)) < 0) {
                throw I2CMasterException("Failed to open device " + mcp2221a_path_ + ": " + strerror(errno));
            }
        } else {
            throw I2CMasterException(MCP2221_NAME " not found or not connected.");
        }
    }

    I2CMaster::~I2CMaster()
    {
        if (fd_ != 0)
            close(fd_);
    }

    void I2CMaster::scan_bus()
    {
        int ret = 0;

        for (int addr = 0x08; addr < I2C_MAX_ADDR; addr++) {

            set_slave_address(addr);

            if ((ret = i2c_smbus_write_quick(fd_, I2C_SMBUS_WRITE)) < 0)
                continue;
            else
                slaves_addr_.push_back(addr);
        }
    }

    vector<int> I2CMaster::get_slaves_addresses(bool force_scan)
    {
        if (slaves_addr_.empty() || force_scan)
            scan_bus();

        return slaves_addr_;
    }

    void I2CMaster::set_slave_address(uint8_t addr)
    {
        if (current_slave_addr_ == addr)
            return;

        if (ioctl(fd_, I2C_SLAVE, addr) < 0) {
            throw I2CMasterException("Failed to set I2C slave address " + to_string(addr) + ": " + strerror(errno));
        }

        current_slave_addr_ = addr;
    }

    void I2CMaster::send(uint8_t slave_addr, uint8_t *data, size_t length) 
    {
        int ret = 0, byte_sent = 0;

        set_slave_address(slave_addr);

        while (byte_sent < length) {
            if ((ret = write(fd_, data, length)) < 0) {
                break;
            }

            byte_sent += ret;
        }

        if (ret < 0) {
            throw I2CMasterException(string("Failed to send data: ") + strerror(errno));
        }
    }

    void I2CMaster::receive(uint8_t slave_addr, uint8_t *data, size_t length)
    {
        int ret = 0, byte_read = 0;

        set_slave_address(slave_addr);

        while (byte_read < length) {
            if ((ret = read(fd_, data, length)) < 0 ) {
                throw I2CMasterException(string("Failed to get data: ") + strerror(errno));
            }
            //EOF
            if (ret == 0)
                break;

            byte_read += ret;
        }
    }

    void I2CMaster::read_register(uint8_t slave_addr, uint8_t reg_addr, uint8_t *data, size_t length)
    {
        set_slave_address(slave_addr);

        if (length < I2C_SMBUS_WORD_DATA) {
            int res = i2c_smbus_read_word_data(fd_, reg_addr);
            data[0] = res >> 8;
            data[1] = res & 0xFF;
            return;
        }

        if (i2c_smbus_write_byte(fd_, reg_addr) < 0) {
            throw I2CMasterException("Failed to write register address: " + string(strerror(errno)));
        }

        receive(slave_addr, data, length);
    }

    void I2CMaster::set_speed(uint32_t speed)
    {
        close(fd_);

        try {
            hid_set_i2c_speed(speed);
        } catch (const HIDException& ex) {
            throw I2CMasterException(ex.what());
        }

        if (!mcp2221a_path_.empty()) {
            if ((fd_ = open(mcp2221a_path_.c_str(), O_RDWR)) < 0) {
                throw I2CMasterException("Failed to open device " + mcp2221a_path_ + ": " + strerror(errno));
            }
        } else {
            throw I2CMasterException(MCP2221_NAME " not found or not connected.");
        }
    }


} //namespace