#include <iostream>

#include "mcp2221a.h"
#include "mcp2221_dll_um.h"

#define MCP2221A_VID        0x04D8
#define MCP2221A_PID        0x00DD
#define MCP2221_INVALID_HANDLE    (void*)(-1)

#define I2C_MAX_ADDR        0x80
#define I2C_MIN_ADDR        0x08
#define I2C_10BIT_ADDRESS   0
#define I2C_7BIT_ADDRESS    1
#define I2C_MAX_RETRIES     5

using namespace std;

namespace MCP2221A {

    static std::string get_error_string(int err_code)
    {
        switch (err_code) {
            case E_ERR_UNKOWN_ERROR:
                return "Unknown error occurred";
            case E_ERR_CMD_FAILED:
                return "Command failed";
            case E_ERR_INVALID_HANDLE:
                return "Invalid device handle";
            case E_ERR_INVALID_PARAMETER:
                return "Invalid parameter";
            case E_ERR_I2C_BUSY:
                return "I2C bus is busy";
            case E_ERR_ADDRESS_NACK:
                return "No ACK received from slave for the address sent";
            case E_ERR_TIMEOUT:
                return "Operation timed out";
            case E_ERR_NO_SUCH_INDEX:
                return "No device with the specified index";
            case E_ERR_CONNECTION_ALREADY_OPENED:
                return "Connection already opened";
            default:
                return "Unknown error code: " + to_string(err_code);
        }
    }

    I2CMaster::I2CMaster(unsigned int speed, unsigned char timeout_ms) : handle_(nullptr), current_slave_addr_(0), i2c_speed_(speed), i2c_timeout_ms_(timeout_ms)
    {
        unsigned int device_count = 0;
        int ret = 0;
        
        if (Mcp2221_GetConnectedDevices(MCP2221A_VID, MCP2221A_PID, &device_count) < 0) {
            throw I2CMasterException("Failed to get connected MCP2221A devices");
        }

        if (device_count == 0) {
            throw I2CMasterException("No MCP2221A devices connected");
        }

        handle_ = Mcp2221_OpenByIndex(MCP2221A_VID, MCP2221A_PID, 0);
        ret = Mcp2221_GetLastError();
        if (ret != E_NO_ERR) {
            handle_ = nullptr;
            throw I2CMasterException("Failed to open MCP2221A device: " + get_error_string(ret));
        }

        set_speed(speed);

        if ((ret = Mcp2221_SetAdvancedCommParams(handle_, i2c_timeout_ms_, I2C_MAX_RETRIES)) < 0) {
            handle_ = nullptr;
            throw I2CMasterException("Failed to set I2C advanced communication parameters: " + get_error_string(ret));
        }
    }   

    // Implementations of pure virtual destructor
    I2CMaster::~I2CMaster() 
    {
        if (handle_) {
            Mcp2221_Close(handle_);
        } else {
            Mcp2221_CloseAll();
        }
    }

    void I2CMaster::scan_bus()
    {
        int ret = 0;
        unsigned char dummy = 0;

        slaves_addr_.clear();

        if ((ret = Mcp2221_SetAdvancedCommParams(handle_, I2C_TIMEOUT_1MS, 2)) < 0) {
            throw I2CMasterException("Failed to set I2C advanced communication parameters: " + get_error_string(ret));
        }

        for (unsigned char addr = I2C_MIN_ADDR; addr < I2C_MAX_ADDR; addr++) {
            try {
                receive(addr, &dummy, 1);
            } catch (const I2CMasterException &e) {
                continue;
            }
            slaves_addr_.push_back(addr);
        }

        if ((ret = Mcp2221_SetAdvancedCommParams(handle_, i2c_timeout_ms_, I2C_MAX_RETRIES)) < 0) {
            throw I2CMasterException("Failed to set I2C advanced communication parameters: " + get_error_string(ret));
        }
    }

    void I2CMaster::set_slave_address(uint8_t addr)
    {
        if (addr < I2C_MIN_ADDR || addr >= I2C_MAX_ADDR) {
            throw I2CMasterException("Invalid slave address: " + to_string(addr));
        }

        current_slave_addr_ = addr;
    }

    void I2CMaster::send(uint8_t slave_addr, uint8_t *data, size_t length) 
    {
        int ret = 0;

        if ((ret = Mcp2221_I2cWrite(handle_, static_cast<unsigned int>(length), slave_addr, I2C_7BIT_ADDRESS, data)) < 0) {
            throw I2CMasterException("I2C write failed: " + get_error_string(ret));
        }
    }

    void I2CMaster::receive(uint8_t slave_addr, uint8_t *data, size_t length) 
    {
        int ret = 0;

        if ((ret = Mcp2221_I2cRead(handle_, static_cast<unsigned int>(length), slave_addr, I2C_7BIT_ADDRESS, data)) < 0) {
            throw I2CMasterException("I2C read failed: " + get_error_string(ret));
        }
    }

    void I2CMaster::read_register(uint8_t slave_addr, uint8_t reg_addr, uint8_t *data, size_t length)
    {
        try {
            send(slave_addr, &reg_addr, 1);
        } catch (const I2CMasterException &e) {
            throw I2CMasterException("Failed to write register address before reading: " + std::string(e.what()));
        }

        try {
            receive(slave_addr, data, length);
        } catch (const I2CMasterException &e) {
            throw I2CMasterException("Failed to read register data: " + std::string(e.what()));
        }
    }

    std::vector<int> I2CMaster::get_slaves_addresses(bool force_scan)
    {
        if (slaves_addr_.empty() || force_scan)
            scan_bus();

        return slaves_addr_;
    }

    void I2CMaster::set_speed(uint32_t speed)
    {
        int ret = 0;

        if ((ret = Mcp2221_SetSpeed(handle_, speed)) < 0) {
            throw I2CMasterException("Failed to set I2C speed: " + get_error_string(ret));
        }
    }

} // namespace MCP2221A