#include <iostream>
#include <map>

extern "C" {
    #include <dirent.h>
    #include <sys/stat.h>
    #include <unistd.h>
    #include <fcntl.h>
    #include <string.h>
    #include <errno.h>
    #include <termios.h>
}

#include "mcp2221a.h"

#define SYSFS_TTY_PATH      "/sys/class/tty/"
#define TTY_BASE_NAME       "ttyACM"
#define MODALIAS            "usb:v04D8p00DD"
#define READ_TIMEOUT        10

using namespace std;

namespace MCP2221A {

    static map<Serial::BAUD, int> baud_map = {
        {Serial::B_300, B300},
        {Serial::B_1200, B1200},
        {Serial::B_2400, B2400},
        {Serial::B_4800, B4800},
        {Serial::B_9600, B9600},
        {Serial::B_19200, B19200},
        {Serial::B_38400, B38400},
        {Serial::B_57600, B57600},
        {Serial::B_115200, B115200},
        {Serial::B_230400, B230400},
        {Serial::B_460800, B460800},
    };

    vector<string> find_ttyacm_devices()
    {
        DIR *target = NULL;
        struct dirent *about;
        struct stat file;
        vector<string> acm_devs;

        target = opendir(SYSFS_TTY_PATH);
        if (!target) {
            goto exit;
        }

        while (about = readdir(target)) {
            size_t found = string(about->d_name).find(TTY_BASE_NAME);
            if (string::npos != found)
                acm_devs.push_back(about->d_name);
        }

        closedir(target);

    exit:
        return acm_devs;
    }

    bool check_tty_modalias(string port)
    {
        int fd = 0, ret = 0;
        char buf;
        string port_modalias_path = SYSFS_TTY_PATH + port + "/device/modalias";
        string modalias = "";

        if ((fd = open(port_modalias_path.c_str(), O_RDONLY)) < 0) {
            cout << "Failed to open file " << port_modalias_path << ": " << strerror(errno) << endl;
            return false;
        }

        while ((ret = read(fd, &buf, 1))) {
            modalias += buf;
        }

        if (ret < 0) {
            cout << "Failed to read file " << port_modalias_path << ": " << strerror(errno) << endl;
            goto error;
        }

        if (modalias.find(MODALIAS) == string::npos)
            goto error;

        close(fd);
        return true;

    error:
        close(fd);
        return false;
    }

    Serial::Serial(string port, BAUD baudrate) : fd_(0), port_(port), baudrate_(baudrate), read_timeout_(READ_TIMEOUT)
    {
        struct termios tty;

        if (!check_tty_modalias(port_)) {
            
            port_ = "";
            vector<string> tty_devs = find_ttyacm_devices();

            for (auto& dev: tty_devs) {
                if (check_tty_modalias(dev)) {
                    port_ = dev;
                    break;
                }
            }
        }

        if (port_.empty())
            throw SerialException(MCP2221_NAME " not found or not connected.");

        // Open port
        if ((fd_ = open(string("/dev/" + port_).c_str(), O_RDWR)) < 0) {
            throw SerialException("Failed to open port " + port_ + ": " + strerror(errno));
        }

        if (tcgetattr(fd_, &tty) < 0) {
            close(fd_);
            throw SerialException(string("Failed to get termios: ") + strerror(errno));
        }

        tty.c_cflag &= ~PARENB; // No parity
        tty.c_cflag &= ~CSTOPB; // One stop bit
        tty.c_cflag &= ~CSIZE; // Clear size bits
        tty.c_cflag |= CS8; // 8bit per byte
        tty.c_cflag &= ~CRTSCTS; // Disable RTS/CTS
        tty.c_cflag |= CREAD | CLOCAL; // Turn on READ & ignore ctrl lines (CLOCAL = 1)

        tty.c_lflag &= ~ICANON; // Disable canonical mode
        tty.c_lflag &= ~ECHO; // Disable echo
        tty.c_lflag &= ~ECHOE; // Disable erasure
        tty.c_lflag &= ~ECHONL; // Disable new-line echo
        tty.c_lflag &= ~ISIG; // Disable interpretation of INTR, QUIT and SUSP

        tty.c_iflag &= ~(IXON | IXOFF | IXANY); // Turn off s/w flow ctrl
        tty.c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|IGNCR|ICRNL); // Disable any special handling of received bytes

        tty.c_oflag &= ~OPOST; // Prevent special interpretation of output bytes (e.g. newline chars)
        tty.c_oflag &= ~ONLCR; // Prevent conversion of newline to carriage return/line feed

        tty.c_cc[VTIME] = READ_TIMEOUT;    // Wait for up to 1s (10 deciseconds), returning as soon as any data is received.
        tty.c_cc[VMIN] = 0;

        // Set in/out baud rate
        cfsetspeed(&tty, baud_map[baudrate_]);
        
        // Apply config to port
        if (tcsetattr(fd_, TCSANOW, &tty) < 0) {
            close(fd_);
            throw SerialException(string("Failed to set termios: ") + strerror(errno));
        }
    }

    Serial::~Serial()
    {
        if (fd_)
            close(fd_);
    }

    void Serial::send(uint8_t *data, size_t length)
    {
        int ret = 0, byte_sent = 0;

        while (byte_sent < length) {
            if ((ret = write(fd_, &data[byte_sent], length - byte_sent)) < 0) {
                break;
            }

            byte_sent += ret;
        }

        if (ret < 0) {
            throw SerialException(string("Failed to send data: ") + strerror(errno));
        }
    }

    void Serial::receive(uint8_t *data, size_t length)
    {
        int ret = 0, byte_read = 0;

        while (byte_read < length) {
            if ((ret = read(fd_, &data[byte_read], length - byte_read)) <= 0) {    
                break;
            }

            byte_read += ret;
        }

        if (ret < 0) {
            throw SerialException(string("Failed to read data: ") + strerror(errno));
        }

        if (ret == 0) {
            if (byte_read == 0)
                throw TimeoutException("Timeout expired and no data was received");
            else if (byte_read < length)
                throw TimeoutException("Timeout expired and " + to_string(byte_read) + "/" + to_string(length) + " bytes were received");
        }
    }

    void Serial::set_baudrate(BAUD baudrate)
    {
        struct termios tty;

        if (baudrate_ == baudrate)
            return;

        if (tcgetattr(fd_, &tty) < 0) {
            throw SerialException(string("Failed to get termios: ") + strerror(errno));
        }

        if (cfsetspeed(&tty, baud_map[baudrate]) < 0) {
            throw SerialException(string("Failed to set speed: ") + strerror(errno));
        }

        // Apply config to port
        if (tcsetattr(fd_, TCSANOW, &tty) < 0) {
            throw SerialException(string("Failed to set termios: ") + strerror(errno));
        }

        baudrate_ = baudrate;
    }

    void Serial::set_timeout(uint8_t timeout)
    {
        struct termios tty;

        if (read_timeout_ == timeout)
            return;

        if (tcgetattr(fd_, &tty) < 0) {
            throw SerialException(string("Failed to get termios: ") + strerror(errno));
        }

        tty.c_cc[VTIME] = timeout; //Max value is 25.5 seconds or 255
        tty.c_cc[VMIN] = 0;
    
        // Apply config to port
        if (tcsetattr(fd_, TCSANOW, &tty) < 0) {
            throw SerialException(string("Failed to set termios: ") + strerror(errno));
        }

        read_timeout_ = timeout;
    }

} // namespace