#include <iostream>
#include "mcp2221a.h"

using namespace std;

/**
 * @brief Test serial capabilities. Connect RX to TX for this to work
 * 
 * @return true 
 * @return false 
 */
bool test_serial(MCP2221A::Serial::BAUD baud = MCP2221A::Serial::B_115200, uint8_t read_timeout = 10)
{
    bool success = true;
    uint8_t tx[] = {'H', 'e', 'l', 'l', 'o', ' ', 'W', 'o', 'r', 'l', 'd', '!'};
    uint8_t rx[sizeof(tx)] = {0};

    MCP2221A::Serial ser;

    try {
        ser.set_timeout(read_timeout);
        ser.set_baudrate(baud);
        ser.send(tx, sizeof(tx));
        ser.receive(rx, sizeof(rx));
    } catch (MCP2221A::SerialException const& ex) {
        cout << ex.what() << endl;
        return false;
    } catch (MCP2221A::TimeoutException const& ex) {
        cout << ex.what() << endl;
        return false;
    }

    for (int i = 0; i < sizeof(rx); i++) {
        if (rx[i] != tx[i]) {
            success = false;
            break;
        }
    }

    return success;
}

/**
 * @brief Test i2c. Connect an SHT30 device
 * 
 * @return true 
 * @return false 
 */
bool test_i2c()
{
    bool success = true, slave_found = false;
    uint8_t sht30_addr = 0x44;
    uint8_t tx[] = {0x37, 0x80};
    uint8_t rx[3] = {0};
    uint8_t rx_expected[] = {0x13, 0xB6, 0xDB};

    MCP2221A::I2CMaster i2c;

    auto slaves = i2c.get_slaves_addresses();
    for (auto& slave: slaves) {
        if (slave == sht30_addr) {
            slave_found = true;
            break;
        }
    }

    if (!slave_found)
        return false;

    // Test read
    try {
        i2c.send(sht30_addr, tx, sizeof(tx));
        i2c.receive(sht30_addr, rx, sizeof(rx));
    } catch (MCP2221A::I2CMasterException const& ex) {
        cout << ex.what() << endl;
        return false;
    }

    for (int i = 0; i < sizeof(rx_expected); i++) {
        if (rx[i] != rx_expected[i]) {
            success = false;
            break;
        }
    }

    return success;
}

int main()
{
    cout << "Testing serial (115200, 1s): ";
    if (!test_serial())
        cout << "Fail" << endl;
    else 
        cout << "Success" << endl;
    
    cout << "Testing serial (9600, 1s): ";
    if (!test_serial(MCP2221A::Serial::B_9600))
        cout << "Fail" << endl;
    else 
        cout << "Success" << endl;

    cout << "Testing serial (9600, 0.1s): ";
    if (!test_serial(MCP2221A::Serial::B_9600, 1))
        cout << "Fail" << endl;
    else 
        cout << "Success" << endl;

    cout << "Testing I2C: ";
    if (!test_i2c())
        cout << "Fail" << endl;
    else 
        cout << "Success" << endl;

    return 0;
}   
