"""This module contains a class to control the MS210 device
"""

from typing import Literal
from dataclasses import dataclass
import serial

_BAUD = 19200
_CHANNELS = ["IR", "R", "B", "G"]
_MAX_VALUE = 1000
_MIN_VALUE = 0


@dataclass
class Channel:
    """Holds channel attributes"""

    name: str
    index: int
    value: int


class MS210InitFailed(Exception):
    """MS210 serial communication failed"""


class MS210:
    """MS210 driver class"""

    def __init__(self, port: str = "/dev/ttyUSB0"):
        """Constructor of the MS210 object

        Parameters
        ----------
        port : str
            COM port of the Serial device. Windows = "COM<X>", Linux = "/dev/ttyUSB<X>"

        Raises
        ------
        MS210InitFailed if device not found
        """

        self._ser = serial.Serial(timeout=2)
        self._ser.port = port
        self._ser.baudrate = _BAUD

        self.channels = []
        for index, channel in enumerate(_CHANNELS):
            self.channels.append(Channel(channel, index, 0))

        try:
            self._ser.open()
        except serial.SerialException as ex:
            raise MS210InitFailed(ex.__str__()) from ex

        # Read current values
        success, msg = self.__get_current_values()
        if not success:
            raise MS210InitFailed(msg)

    def __del__(self):
        """Destructor"""
        for channel in _CHANNELS:
            self.set_value(channel, 0)

        self._ser.close()

    def __check_limits(self, value: int) -> bool:
        """Check that the value is between the limits

        Parameters
        ----------
        value : int
            Value to check

        Returns
        -------
        False if value is out of bound, True otherwise
        """

        if value > _MAX_VALUE:
            return False

        if value < _MIN_VALUE:
            return False

        return True

    def __get_current_values(self) -> tuple[bool, str]:
        """Read the current values of each channel

        Returns
        -------
        tuple : bool, str
            success, error message if not success
        """

        buf = ""

        for channel in self.channels:
            try:
                self._ser.write(f"R{channel.index}".encode())
            except serial.SerialException as ex:
                return (False, ex.__str__())

            try:
                buf = self._ser.read_until(size=8)
            except serial.SerialException as ex:
                return (False, ex.__str__())

            if len(buf) < 1:
                return (False, "No bytes received. Timeout expired")

            channel.value = int(buf[2:].decode().strip())

        return (True, "OK")

    def get_value(self, channel: Literal["IR", "R", "B", "G"]) -> int:
        """Get the value of a channel

        Parameters
        ----------
        channel : str
            The channel name

        Returns
        -------
        The current value of the channel
        """

        return [x for x in self.channels if x.name == channel][0].value

    def set_value(
        self, channel: Literal["IR", "R", "B", "G"], value: int = 0
    ) -> tuple[bool, str]:
        """Set the value of a channel

        Parameters
        ----------
        channel : str
            The name of the channel
        value : int
            The value to set the channel to

        Returns
        -------
        tuple: bool, str
            success, an error msg in not success
        """

        if not self.__check_limits(value):
            return (
                False,
                f"Value {value} is out of bound. {_MIN_VALUE} < value < {_MAX_VALUE}",
            )

        _channel = [x for x in self.channels if x.name == channel][0]
        if value == _channel.value:
            return (True, "OK")

        cmd = f"S{_channel.index}{value:04d}"

        try:
            self._ser.write(cmd.encode())
        except serial.SerialException as ex:
            return (False, ex.__str__())
        try:
            buf = self._ser.read(8)
        except serial.SerialException as ex:
            return (False, ex.__str__())

        _channel.value = int(buf[2:].decode().strip())

        return (True, "OK")
