import sys
from typing import Literal
from PyQt6.QtGui import QCloseEvent

from PyQt6.QtWidgets import QApplication, QMainWindow, QMessageBox
from app.ms210_ui import Ui_MainWindow

from ms210.driver import MS210, MS210InitFailed
import serial.tools.list_ports

class Window(QMainWindow, Ui_MainWindow):

    def __init__(self, parent=None) -> None:
        super().__init__(parent)
        self.setupUi(self)
        self._ms210 = None
        self.__get_ports()
        self.__setup_signals()

    def __get_ports(self):
 
        self.port_cb.addItem("None")
        ports = serial.tools.list_ports.comports()

        for port in ports:
            if "USB" in port.name:
                self.port_cb.addItem(port.name)

    def __setup_signals(self):
        self.port_cb.currentIndexChanged.connect(self.port_select_cb)
        
        # IR
        self.ir_slider.sliderReleased.connect(self.ir_slider_cb)
        self.ir_sb.valueChanged.connect(self.ir_sb_cb)

        # Red 
        self.red_slider.sliderReleased.connect(self.red_slider_cb)
        self.red_sb.valueChanged.connect(self.red_sb_cb)

        # Green
        self.green_slider.sliderReleased.connect(self.green_slider_cb)
        self.green_sb.valueChanged.connect(self.green_sb_cb)

        # Blue
        self.blue_slider.sliderReleased.connect(self.blue_slider_cb)
        self.blue_sb.valueChanged.connect(self.blue_sb_cb)

    def __set_value(self, channel : Literal["IR", "R", "B", "G"], value: int = 0) -> bool:

        if self._ms210 is not None:
            success, msg = self._ms210.set_value(channel, value)
            if not success:
                self.error_msg(f"Failed to set {channel}: {msg}")
                return False
        else:
            self.error_msg("Please select a serial port")
            return False
        
        return True

    def __init_values(self):
        
        ir = self._ms210.get_value("IR")
        self.ir_sb.setValue(ir)
        self.ir_slider.setValue(ir)
        
        red = self._ms210.get_value("R")
        self.red_sb.setValue(red)
        self.red_slider.setValue(red)
        
        green = self._ms210.get_value("G")
        self.green_sb.setValue(green)
        self.green_slider.setValue(green)
        
        blue = self._ms210.get_value("B")
        self.blue_sb.setValue(blue)
        self.blue_slider.setValue(blue)

    def port_select_cb(self):

        if self.port_cb.currentText() == "None":
            self._ms210 = None
            return

        try:
            self._ms210 = MS210(f"/dev/{self.port_cb.currentText()}")
        except MS210InitFailed as ex:
            self.error_msg(f"Failed to connect to MS210: {ex}")
            self._ms210 = None
            return

        self.__init_values()

    def ir_slider_cb(self):
        if not self.__set_value("IR", self.ir_slider.value()):
            self.ir_slider.setValue(self.ir_sb.value())
        else:
            self.ir_sb.value(self.ir_slider.value())
        
    def ir_sb_cb(self):
        if not self.__set_value("IR", self.ir_sb.value()):
            self.ir_slider.setValue(self.ir_slider.value())
        else:
            self.ir_sb.setValue(self.ir_sb.value())
    
    def red_slider_cb(self):
        if not self.__set_value("R", self.red_slider.value()):
            self.red_slider.setValue(self.red_sb.value())
        else:
            self.red_sb.setValue(self.red_slider.value())

    def red_sb_cb(self):
        if not self.__set_value("R", self.red_sb.value()):
            self.red_sb.setValue(self.red_slider.value())
        else:
            self.red_slider.setValue(self.red_sb.value())

    def green_slider_cb(self):
        if not self.__set_value("G", self.green_slider.value()):
            self.green_slider.setValue(self.green_sb.value())
        else:
            self.green_sb.setValue(self.green_slider.value())

    def green_sb_cb(self):
        if not self.__set_value("G", self.green_sb.value()):
            self.green_sb.setValue(self.green_slider.value())
        else:
            self.green_slider.setValue(self.green_sb.value())

    def blue_slider_cb(self):
        if not self.__set_value("B", self.blue_slider.value()):
            self.blue_slider.setValue(self.blue_sb.value())
        else:
            self.blue_sb.setValue(self.blue_slider.value())

    def blue_sb_cb(self):
        if not self.__set_value("B", self.blue_sb.value()):
            self.blue_sb.setValue(self.blue_slider.value())
        else:
            self.blue_slider.setValue(self.blue_sb.value())

    def error_msg(self, msg : str):
        QMessageBox.critical(None, "Error", msg)

    def closeEvent(self, a0: QCloseEvent) -> None:
        self._ms210 = None
        return super().closeEvent(a0)      

def main():

    app = QApplication(sys.argv)
    win = Window()
    win.show()
    sys.exit(app.exec())

if "__main__" == __name__:
    main()