#!/bin/bash

SCRIPTPATH="$( cd "$(dirname "$0")" >/dev/null 2>&1 ; pwd -P )"

PROJECT_PATH=""
BOARD=rpi_pico
OPENOCD="/usr/local/bin/openocd"
OPENOCD_PATH="/usr/local/share/openocd/scripts"
OPENOCD_ADAPTER=cmsis-dap

help()
{
    echo "Build zephyr project"
    echo
    echo "Syntax: $0 [-b|c|d|f|h|] <args>"
    echo "[ -b PATH ]   Build project. Path to project folder"
    echo "[ -c ]        Clean"
    echo "[ -d ]        Launch debugger"
    echo "[ -f ]        Flash board"
    echo "[ -h ]        Help"
}

build=0
clean=0
debug=0
flash=0

if [ $# -eq 0 ]; then
    echo "Error: missing arguments"
    help
    exit 0
else
    while getopts "b:cdfh" opt; do
        case $opt in
            b)
                build=1
                PROJECT_PATH=${OPTARG}
                ;;
            c)
                clean=1
                ;;
            d)
                debug=1
                ;;
            
            f)
                flash=1
                ;;
            h)
                help
                exit 1
                ;;
            \?)
                help
                exit 1
                ;;
            :)
                echo "Option -$OPTARG requires an argument." >&2
                exit 1
                ;;
        esac
    done
fi

if [ ! -d $PROJECT_PATH ]; then
    echo "${PROJECT_PATH} does not exist"
    exit 1
fi

source .venv/bin/activate

if [ $clean == 1 ]; then
    west build -t clean
fi

if [ $build == 1 ]; then
    echo "Building project ${PROJECT_PATH}"
    export ZEPHYR_BASE=$SCRIPTPATH/zephyr 
    west build -p auto -b $BOARD $PROJECT_PATH -- -DOPENOCD=$OPENOCD -DOPENOCD_DEFAULT_PATH=$OPENOCD_PATH -DRPI_PICO_DEBUG_ADAPTER=$OPENOCD_ADAPTER
fi

if [ $flash == 1 ]; then
    west flash
fi

if [ $debug == 1 ]; then
    sudo openocd -f interface/${OPENOCD_ADAPTER}.cfg -f target/rp2040.cfg -c "adapter speed 5000"
fi

deactivate