#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include "rsvp.h"

#define ERR_STR_SIZE            256
static char error_str[ERR_STR_SIZE] = {0};

RSVP_TYPE get_type(char *type)
{
    RSVP_TYPE type_ = STR;

    for (int i = 0; i < TYPE_COUNT; i++) {
        int n = i > 2 ? 4 : 3;

        if (strncmp(type, rsvp_type_str[i], n) == 0) {
            type_ = (RSVP_TYPE)i;
        }
    }

    return type_;
}

int count_tokens(char *str, char delimiter)
{
    int counter = 0;
    char *begin = str;

    while ((begin = strchr(begin, delimiter)) != NULL) {
        counter++;
        begin++;
    }

    return counter;  
}

void set_error_string(char* format, ...)
{
    memset(error_str, 0, ERR_STR_SIZE);
    va_list ap;
    va_start(ap, format);
    vsnprintf(error_str, ERR_STR_SIZE, format, ap);
    va_end(ap);
}

rsvp_vecd_elem_t* get_vecd_elem(char *token)
{
    char *value = NULL, *begin = token;
    rsvp_vecd_elem_t* elem = NULL;
    int index = 0;
    size_t name_len = 0;

    while ((*token != '\0') && (value == NULL)) {

        switch (*token) {
            case '=':
                value = ++token;
                break;
            default:
                index++;
                token++;
                break;
        }
    }

    elem = (rsvp_vecd_elem_t*)malloc(sizeof(rsvp_vecd_elem_t));
    if (!elem) {
        goto exit;
    }

    name_len = index + 1;
    elem->name = (char*)malloc(name_len);
    if (!elem->name) {
        goto free_elem;
    }

    memset(elem->name, 0, name_len);
    memcpy(elem->name, begin, name_len - 1);
    
    if ((elem->value = strtod(value, NULL)) == 0.0) {
        set_error_string("Failed to convert %s to double", value);
        goto free_name;
    }

    return elem;

free_name:
    free(elem->name);
free_elem:
    free(elem);
exit:
    return NULL;
}

rsvp_vecd_elem_t* rsvp_var_get_vecd_elem(rsvp_var_t* vecd, int index)
{
    if (index >= vecd->len) {
        rsvp_error_str("Index out of range for value %s", vecd->name);
        return NULL;
    }
    return (rsvp_vecd_elem_t*)vecd->value.vecd + index;
}

rsvp_var_t* process_args(char *arg)
{
    char *begin = arg, *name = NULL, *value = NULL;
    RSVP_TYPE type = STR;
    int index = 0, type_start = 0;
    rsvp_var_t* var = NULL;
    size_t name_len = 0;

    // Search for the variable name
    while ((*arg != '\0') && (value == NULL))  {

        switch (*arg)
        {
        case '(':
            name_len = index + 1;
            name = (char*)malloc(name_len);
            memset(name, 0, name_len);
            memcpy(name, begin, name_len - 1);
            type_start = index + 1;
            break;

        case ')':
        {
            size_t type_len = (index - type_start) + 1;
            char type_str[type_len];
            memcpy(type_str, &begin[type_start], type_len);
            type = get_type(type_str);
            break;
        }
        case '=':
            // We found the variable value
            if (name == NULL) {
                name_len = index + 1;
                name = (char*)malloc(name_len);
                memset(name, 0, name_len);
                memcpy(name, begin, name_len - 1);
            }
            value = ++arg;
            break;

        default:
            index++;
            break;
        }

        arg++;
    }

    switch (type) {

        case INT:
        {
            int value_int = strtol(value, NULL, 0);
            var = rsvp_var_create_int(name, value_int);
            break;
        }

        case DBL:
        {
            double value_dbl = strtod(value, NULL);
            var = rsvp_var_create_double(name, value_dbl);
            break;
        }

        case BOOL:
        {
            int val = 0;
            if (strcmp(value, "TRUE") == 0) {
                val = 1;
            }
            var = rsvp_var_create_bool(name, val);
            break;
        }

        case VECD:
        {
            char *token = NULL, delimiter = '=';
            int elem_count = count_tokens(value, delimiter);
            var = rsvp_var_create_vecd(name);

            delimiter = ',';
            token = strtok(++value, &delimiter);
            for (int i = 0; i < elem_count; i++) {
                rsvp_vecd_elem_t *elem = NULL;
                elem = get_vecd_elem(token);
                if (!elem) {
                    printf("Error: %s\n", error_str);
                    continue;
                }
                rsvp_var_vecd_add_elem(var, elem->name, elem->value);
                free(elem->name);
                free(elem);
                token = strtok(NULL, &delimiter);
            }

            break;
        }

        case VECS:
        {
            char *token = NULL, delimiter = '\t';
            int elem_count = count_tokens(value, delimiter) + 1;
            var = rsvp_var_create_vecs(name);

            token = strtok(value, &delimiter);
            for (int i = 0; i < elem_count; i++) {
                rsvp_var_vecs_add_elem(var, token);
                token = strtok(NULL, &delimiter);
            }
            break;
        }


        default:
            var = rsvp_var_create_string(name, value);
            break;

    }

#ifdef DEBUG
    rsvp_var_print(var);
#endif


    if (!var) {
        set_error_string("Failed to add arg %s\n", name);
        free(name);
        return NULL;
    }

    free(name);
    return var;
}

rsvp_data_t *rsvp_data_parse(char* data)
{
    char *token = NULL, *raw_data = NULL;
    char **args = NULL;
    char delimiter = ';';
    size_t index = 0, token_len = 0;

    rsvp_data_t *rsvp_data = (rsvp_data_t*)malloc(sizeof(rsvp_data_t));
    if (!rsvp_data) {
        return NULL;
    }

    raw_data = malloc(strlen(data) + 1);
    memset(raw_data, 0, strlen(data) + 1);
    memcpy(raw_data, data, strlen(data));

    memset(rsvp_data, 0, sizeof(rsvp_data_t));
    rsvp_data->args_count = count_tokens(raw_data, delimiter);
    args = (char**)malloc(rsvp_data->args_count * sizeof(char*));

    // Get command string
    token = strtok(raw_data, &delimiter);
    token_len = strlen(token) + 1;
    rsvp_data->cmd = (char*)malloc(token_len);
    memset(rsvp_data->cmd, 0, token_len);
    memcpy(rsvp_data->cmd, token, token_len - 1);

    // Process args
    while (token != NULL) {
        token = strtok(NULL, &delimiter);
        if (!token) {
            // No more tokens
            break;
        }

        token_len = strlen(token) + 1;

#ifdef DEBUG
        printf("*** DBG *** token[%d]: %s\n", rsvp_data->args_count, token);
#endif
        args[index] = (char*)malloc(token_len);
        memset(args[index], 0, token_len);
        memcpy(args[index++], token, token_len - 1);
    }

    rsvp_data->args = (rsvp_var_t**)malloc(rsvp_data->args_count * sizeof(rsvp_var_t*));
    if (!rsvp_data->args) {
        return NULL;
    }

#ifdef DEBUG
    rsvp_data_print(rsvp_data);
#endif

    for(int i = 0; i < rsvp_data->args_count; i++) {
        rsvp_data->args[i] = process_args(args[i]);
        if (!rsvp_data->args[i]) {
            break;
        }
    }

    for (int i = 0; i < rsvp_data->args_count; i++) {
        free(args[i]);
    }

    free(args);
    free(raw_data);

    return rsvp_data;
}

rsvp_data_t* rsvp_data_create(char* cmd)
{
    rsvp_data_t *data = NULL;
    size_t len = strlen(cmd) + 1;

    data = (rsvp_data_t*)malloc(sizeof(rsvp_data_t));
    if (!data) {
        set_error_string("Failed to allocate memory for data %s\n", cmd);
        return NULL;
    }
    memset(data, 0, sizeof(rsvp_data_t));

    data->cmd = (char*)malloc(len);
    if (!data->cmd) {
        set_error_string("Failed to allocate memory for data command %s\n", cmd);
        free(data);
        return NULL;
    }

    memset(data->cmd, 0, len);
    memcpy(data->cmd, cmd, len - 1);

    return data;
}

int rsvp_data_add_var(rsvp_data_t* data, rsvp_var_t* var)
{
    data->args_count += 1;

    data->args = (rsvp_var_t**)realloc(data->args, sizeof(rsvp_data_t*) * data->args_count);
    if (!data->args) {
        set_error_string("Failed to allocate memory for new variable %s\n", var->name);
        data->args_count -= 1;
        return -1;
    }

    data->args[data->args_count - 1] = var;

    return NO_ERROR;
}

int rsvp_data_add_vars(rsvp_data_t* data, unsigned int count, ...)
{
    va_list args;
    va_start(args, count);

    for (unsigned int i = 0; i < count; i++) {
        rsvp_var_t* var = va_arg(args, rsvp_var_t*);
        if (rsvp_data_add_var(data, var) < 0) {
            return -1;
        }
    }

    va_end(args);
    return NO_ERROR;
}

int count_char_in_int(int value)
{
    // Take into account the sign
    int chars = value < 0 ? 1 : 0;
    int modulo = 0;

    if (value == 0)
        return 1;
    
    value = abs(value);

    while (value > 0) {
        modulo = value % 10;

        if (modulo == 0)
            value /= 10;
        else
            value = (value - modulo) / 10;
        
        chars++;
    }

    return chars;
}

int count_chars_in_double(double value, int precision)
{  
    // Start at precision. Plus one to account for the dot
    int chars = precision + 1;
    int int_value = (int)value;

    // Account for negative 0
    if (value > -1.0 && value < 0.0)
        chars += 1;

    return count_char_in_int(int_value) + chars;
}

char* rsvp_data_to_string(rsvp_data_t* data, int double_precision)
{
    char *str = NULL;
    size_t len = strlen(data->cmd) + 1;
    char dbl_format[] = "%.5f";
    
    str = (char*)malloc(len);
    if (!str) {
        set_error_string("Allocation for command failed");
        return NULL;
    }
    memset(str, 0, len);
    memcpy(str, data->cmd, len - 1);

    for (unsigned int i = 0; i < data->args_count; i++) {
        rsvp_var_t *arg = data->args[i];
        const char *type = rsvp_type_str[arg->type];
        char *arg_value = NULL, *arg_name = NULL;
        size_t value_len = 0, arg_len = 0;
        int ret = 0, index = 0;
        //       ";      NAME            (       INT         )=\0"
        arg_len = 1 + strlen(arg->name) + 1 + strlen(type) + 3;
        arg_name = (char*)malloc(arg_len);
        if (snprintf(arg_name, arg_len, ";%s(%s)=", arg->name, type) != (arg_len - 1))
            ret = -1;
        
        switch (arg->type) {
            case INT:
                value_len = count_char_in_int(arg->value.i) + 1;
                arg_value = (char*)malloc(value_len);
                if (snprintf(arg_value, value_len, "%d", arg->value.i) != (value_len - 1))
                    ret = -1;
                break;
            case DBL:
                value_len = count_chars_in_double(arg->value.d, double_precision) + 1;
                arg_value = (char*)malloc(value_len);
                if (snprintf(arg_value, value_len, dbl_format, arg->value.d) != (value_len - 1))
                    ret = -1;
                break;

            case BOOL:
                char *value = arg->value.b > 0 ? "TRUE" : "FALSE";
                value_len = strlen(value) + 1;
                arg_value = (char*)malloc(value_len);
                if (snprintf(arg_value, value_len, "%s", value) != (value_len - 1))
                    ret = -1;
                break;

            case STR:
                value_len = strlen(arg->value.s) + 1;
                arg_value = (char*)malloc(value_len);
                if (snprintf(arg_value, value_len, "%s", arg->value.s) != value_len - 1)
                    ret = -1;
                break;

            case VECD:
                // []
                value_len = 2;
                for (int i = 0; i < arg->len; i++) {
                    rsvp_vecd_elem_t* elem = rsvp_var_get_vecd_elem(arg, i);
                    if (!elem) {
                        ret = -1;
                        break;
                    }
                    // NAME + = + DOUBLE + , or \0
                    value_len += strlen(elem->name) + 1 + count_chars_in_double(elem->value, double_precision) + 1;
                }
                
                arg_value = (char*)malloc(value_len);
                memset(arg_value, 0, value_len);
                arg_value[index++] = '[';

                for (int i = 0; i < arg->len; i++) {
                    rsvp_vecd_elem_t* elem = rsvp_var_get_vecd_elem(arg, i);
                    size_t dbl_len = count_chars_in_double(elem->value, double_precision) + 1;
                    char* dbl_val = (char*)malloc(dbl_len + 1);
                    snprintf(dbl_val, dbl_len, dbl_format, elem->value);
                    strcat(arg_value, elem->name);
                    index += strlen(elem->name);
                    arg_value[index++] = '=';
                    strcat(arg_value, dbl_val);
                    index += strlen(dbl_val);
                    arg_value[index++] = ',';
                }
                arg_value[index - 1] = ']';
                break;

            case VECS:
                for (int i = 0; i < arg->len; i++) {
                    value_len += strlen(arg->value.vecs[i]) + 1;
                }

                arg_value = (char*)malloc(value_len);
                memset(arg_value, 0, value_len);

                for (int i = 0; i < arg->len; i++) {
                    char *str = arg->value.vecs[i];
                    strcat(arg_value, str);
                    index += strlen(str);
                    arg_value[index++] = '\t';
                }
                arg_value[index-1] = '\0';
                break;
        }

        if (ret < 0) {
            //Skip this arg
            continue;
        }

        str = (char*)realloc(str, len + strlen(arg_name) + strlen(arg_value));
        str = strcat(str, arg_name);
        str = strcat(str, arg_value);
        len = strlen(str) + 1;

        free(arg_value);
        free(arg_name);
    }

    return str;
}

void rsvp_var_print(rsvp_var_t *var)
{
    printf("Name: %s, type: %s, value: ", var->name, rsvp_type_str[var->type]);

    switch (var->type)
    {
    case INT:
        printf("%d\n", var->value.i);
        break;
    
    case DBL:
        printf("%f\n", var->value.d);
        break;

    case BOOL:
        printf("%d\n", var->value.b);
        break;
    
    case VECD:
        printf("[");
        for (int i = 0; i < var->len; i++) {
            rsvp_vecd_elem_t* elem = rsvp_var_get_vecd_elem(var, i);
            printf("%s=%f", elem->name, elem->value);
            if (i < var->len - 1) {
                printf(",");
            }
            elem++;
        }
        printf("]\n");
        break;
    
    case VECS:
        for (int i = 0; i < var->len; i++) {
            printf("%s\t", var->value.vecs[i]);
        }
        printf("\n");
        break;

    default:
        printf("%s\n", var->value.s);
        break;
    }
}

void rsvp_data_print(rsvp_data_t *data)
{
    printf("Command: %s, args [%d]:\n", data->cmd, data->args_count);

    for(int i = 0; i < data->args_count; i++) {
        printf("ARG[%d] => ", i);
        rsvp_var_print(data->args[i]);
    }
}

void rsvp_data_free(rsvp_data_t *data)
{
    if (data != NULL) {
        for (int i = 0; i < data->args_count; i++) {
            rsvp_var_free(data->args[i]);
        }

        if (data->args)
            free(data->args);
        
        free(data->cmd);
        free(data);
    }
}

void rsvp_var_free(rsvp_var_t *var) 
{
    if (var) {
        switch (var->type)
        {
        case VECD:
            for (int i = 0; i < var->len; i++) {
                rsvp_vecd_elem_t* elem = rsvp_var_get_vecd_elem(var, i);
                free(elem->name);
            }
            free(var->value.vecd);
            break;

        case VECS:
            for (int i = 0; i < var->len; i++) {
                free(var->value.vecs[i]);
            }
            free(var->value.vecs);
            break;

        case STR:
            free(var->value.s);
            break;
        
        default:
            break;
        }

        free(var->name);
        free(var);
    }
}

rsvp_var_t* rsvp_var_init(char *name)
{
    size_t len = strlen(name) + 1;
    rsvp_var_t *var = (rsvp_var_t*)malloc(sizeof(rsvp_var_t));
    if (!var) {
        set_error_string("Allocation of variable %s failed", name);
        return NULL;
    }
    memset(var, 0, sizeof(rsvp_var_t));
    
    var->name = (char*)malloc(len);
    if (!var->name) {
        set_error_string("Allocation of variable %s name's failed", name);
        free(var);
        return NULL;
    }
    memset(var->name, '\0', len);
    memcpy(var->name, name, len - 1);

    return var;
}

rsvp_var_t* rsvp_var_create_int(char *name, int value)
{
    rsvp_var_t *var = rsvp_var_init(name);
    if (!var)
        return NULL;

    var->type = INT;
    var->value.i = value;

    return var;
}

rsvp_var_t* rsvp_var_create_double(char *name, double value)
{
    rsvp_var_t *var = rsvp_var_init(name);
    if (!var)
        return NULL;

    var->type = DBL;
    var->value.d = value;

    return var;
}

rsvp_var_t* rsvp_var_create_bool(char *name, int value)
{
    rsvp_var_t *var = rsvp_var_init(name);
    if (!var)
        return NULL;

    var->type = BOOL;
    var->value.b = value;

    return var;
}

rsvp_var_t* rsvp_var_create_string(char *name, char* value)
{
    size_t len = strlen(value) + 1;
    rsvp_var_t *var = rsvp_var_init(name);
    if (!var)
        return NULL;

    var->type = STR;
    var->value.s = (char*)malloc(len);
    if (!var->value.s) {
        return NULL;
    }
    memset(var->value.s, 0, len);
    memcpy(var->value.s, value, len - 1);

    return var;
}

rsvp_var_t* rsvp_var_create_vecd(char *name)
{
    rsvp_var_t *var = rsvp_var_init(name);
    if (!var)
        return NULL;
        
    var->type = VECD;

    return var;
}

int rsvp_var_vecd_add_elem(rsvp_var_t* vecd, char* name, double value)
{
    size_t name_len = strlen(name) + 1;
    
    if (vecd->type != VECD) {
        set_error_string("Failed to add element to variable %s because is of type %s instead of VECD.", vecd->name, rsvp_type_str[vecd->type]);
        return -1;
    }

    vecd->len += 1;

    vecd->value.vecd = (rsvp_vecd_elem_t*)realloc(vecd->value.vecd, vecd->len * sizeof(rsvp_vecd_elem_t));
    if (!vecd->value.vecd) {
        set_error_string("Failed to augment the size of %s", vecd->name);
        vecd->len -= 1;
        return -1;
    }

    vecd->value.vecd[vecd->len - 1].name = (char*)malloc(name_len);
    if (!vecd->value.vecd[vecd->len - 1].name) {
        set_error_string("Failed to allocate name for element %s", name);
        vecd->len -=1;
        vecd->value.vecd = (rsvp_vecd_elem_t*)realloc(vecd->value.vecd, vecd->len * sizeof(rsvp_vecd_elem_t));
        return -1;
    }

    memset(vecd->value.vecd[vecd->len - 1].name, 0, name_len);
    memcpy(vecd->value.vecd[vecd->len - 1].name, name, name_len - 1);
    vecd->value.vecd[vecd->len - 1].value = value;

    return NO_ERROR;
}

rsvp_var_t* rsvp_var_create_vecs(char *name)
{
    rsvp_var_t *var = rsvp_var_init(name);
    if (!var)
        return NULL;
        
    var->type = VECS;

    return var;
}

int rsvp_var_vecs_add_elem(rsvp_var_t* vecs, char *value)
{
    size_t value_len = strlen(value) + 1;

    if (vecs->type != VECS) {
        set_error_string("Failed to add element to variable %s because is of type %s instead of VECS.", vecs->name, rsvp_type_str[vecs->type]);
        return -1;
    }

    vecs->len += 1;

    vecs->value.vecs = (char**)realloc(vecs->value.vecs, vecs->len * sizeof(char*));
    if (!vecs->value.vecs) {
        vecs->len -= 1;
        return -1;
    }

    vecs->value.vecs[vecs->len - 1] = (char*)malloc(value_len);
    if (!vecs->value.vecs[vecs->len - 1]) {
        return -1;
    }

    memset(vecs->value.vecs[vecs->len - 1], 0, value_len);
    memcpy(vecs->value.vecs[vecs->len - 1], value, value_len - 1);

    return NO_ERROR;
}

char* rsvp_var_vecs_get_elem(rsvp_var_t* vecs, int index)
{
    if (index >= vecs->len) {
        rsvp_error_str("Index out of range for value %s", vecs->name);
        return NULL;
    }

    return vecs->value.vecs[index];
}

const char* rsvp_error_str()
{
    return (const char*)&error_str;
}