#ifndef RSVP_H
#define RSVP_H

#define TYPE_COUNT  6
#define NO_ERROR    0

typedef enum  {
    STR = 0,
    INT,
    DBL,
    BOOL,
    VECD,
    VECS,
} RSVP_TYPE;

/**
 * @brief Conversion of the RSVP_TYPE enum to string
 * 
 */
const static char rsvp_type_str [TYPE_COUNT][5] = {
    "STR\0",
    "INT\0",
    "DBL\0",
    "BOOL",
    "VECD",
    "VECS"
};

/**
 * @brief Array of double element struct
 * 
 */
struct rsvp_vecd_elem {
    char *name;
    double value;
};
typedef struct rsvp_vecd_elem rsvp_vecd_elem_t;

/**
 * @brief Define the possible values of RSVP variable value
 * 
 */
typedef union
{
    int i;
    double d;
    int b;
    char *s;
    rsvp_vecd_elem_t* vecd;
    char** vecs;

} value_t;

/**
 * @brief Generic RSVP variable structure
 * 
 */
struct rsvp_var {
    char *name;
    int len;
    RSVP_TYPE type;
    value_t value;
};
typedef struct rsvp_var rsvp_var_t;

/**
 * @brief RSVP data struct
 * 
 */
struct rsvp_data {
    char *cmd;
    rsvp_var_t **args;
    unsigned int args_count;
};
typedef struct rsvp_data rsvp_data_t;

/**
 * @brief Parse raw RSVP data. Remember to free the memory using rsvp_data_free
 * 
 * @param data Data to parse
 * @return rsvp_data_t* Pointer to the parsed data
 */
extern rsvp_data_t* rsvp_data_parse(char* data);

/**
 * @brief Create a new RSVP data struct. Remember to free the memory using rsvp_data_free 
 * 
 * @param cmd RSVP data command
 * @return rsvp_data_t* RSVP data struct
 */
extern rsvp_data_t* rsvp_data_create(char* cmd);

/**
 * @brief Add an argument to the RSVP data struct
 * 
 * @param data Pointer of the RSVP data struct to add the argument to
 * @param var RSVP argument to be added
 * @return int 0 on success, -1 on failure. Use rsvp_error_string to get more information
 */
extern int rsvp_data_add_var(rsvp_data_t* data, rsvp_var_t* var);

/**
 * @brief Add multiple arguments to the RSVP data struct
 * 
 * @param data Pointer of the RSVP data struct to add the argument to
 * @param count Number of arguments to be added
 * @param ... RSVP argument to be added (<= count). Type rsvp_var_t*
 * @return int 0 on success, -1 on failure. Use rsvp_error_string to get more information
 */
extern int rsvp_data_add_vars(rsvp_data_t* data, unsigned int count, ...);

/**
 * @brief Convert rsvp_data_t to a string of raw RSVP data. Remember to free the returned string
 * 
 * @param data Data to convert
 * @param double_precision Number of decimals for for double values
 * @return char* String of raw RSVP data
 */
extern char* rsvp_data_to_string(rsvp_data_t* data, int double_precision);

/**
 * @brief Print the RSVP data struct to stdout
 * 
 * @param data Data to print
 */
extern void rsvp_data_print(rsvp_data_t* data);

/**
 * @brief Free allocated memory used by data
 * 
 * @param data Data to free 
 */
extern void rsvp_data_free(rsvp_data_t *data);

// Variable creation
/**
 * @brief Create a new RSVP integer variable
 * 
 * @param name Variable name
 * @param value Variable integer value
 * @return rsvp_var_t* Pointer to the created variable
 */
extern rsvp_var_t* rsvp_var_create_int(char *name, int value);

/**
 * @brief Create a new RSVP double variable
 * 
 * @param name Variable name
 * @param value Variable double value
 * @return rsvp_var_t* Pointer to the created variable
 */
extern rsvp_var_t* rsvp_var_create_double(char *name, double value);

/**
 * @brief Create a new RSVP boolean variable
 * 
 * @param name Variable name
 * @param value Variable boolean value
 * @return rsvp_var_t* Pointer to the created variable
 */
extern rsvp_var_t* rsvp_var_create_bool(char *name, int value);

/**
 * @brief Create a new RSVP string variable
 * 
 * @param name Variable name
 * @param value Variable string value
 * @return rsvp_var_t* Pointer to the created variable
 */
extern rsvp_var_t* rsvp_var_create_string(char *name, char* value);

/**
 * @brief Create a new empty array of double variable
 * 
 * @param name Variable name
 * @return rsvp_var_t* Pointer to the created variable
 */
extern rsvp_var_t* rsvp_var_create_vecd(char *name);

/**
 * @brief Add an element to a array of double variable
 * 
 * @param vecd Array of double variable to add the element to
 * @param name Element name
 * @param value Element value
 * @return int 0 on success, -1 on failure. Use rsvp_error_string to get more information
 */
extern int rsvp_var_vecd_add_elem(rsvp_var_t* vecd, char* name, double value);

/**
 * @brief Get a specific element from an array of double variable
 * 
 * @param var Array of double
 * @param index Element index
 * @return rsvp_vecd_elem_t* Element
 */
extern rsvp_vecd_elem_t* rsvp_var_get_vecd_elem(rsvp_var_t* vecd, int index);

/**
 * @brief Create a new array of string variable
 * 
 * @param name Variable name
 * @return rsvp_var_t* Pointer to the created variable
 */
extern rsvp_var_t* rsvp_var_create_vecs(char *name);

/**
 * @brief Add a string to an array of strings
 * 
 * @param vecs Array of strings
 * @param value String to add
 * @return int 0 on success, -1 on failure. Use rsvp_error_string to get more information
 */
extern int rsvp_var_vecs_add_elem(rsvp_var_t* vecs, char* value);

/**
 * @brief Get a specific element from an array of string variable
 * 
 * @param vecs Array of strings
 * @param index Element index
 * @return char* String 
 */
extern char* rsvp_var_vecs_get_elem(rsvp_var_t* vecs, int index);

/**
 * @brief Print a RSVP variable to stdout
 * 
 * @param var Variable to print
 */
extern void rsvp_var_print(rsvp_var_t *var);

/**
 * @brief Free an allocated RSVP variable
 * 
 * @param var Variable to free
 */
extern void rsvp_var_free(rsvp_var_t *var);

/**
 * @brief Get the current error string
 * 
 * @return const char* Error string
 */
extern const char* rsvp_error_str();

#endif //RSVP_H