#include "rsvppp.h"
#include <iostream>

using namespace std;

namespace RSVP {

    Packet::Packet(string command) : command_(command) { }

    Packet::~Packet()
    {
        for (size_t i = 0; i < args_.size(); i++) {
            rsvp_var_t *arg = args_[i];
            rsvp_var_free(arg);
        }
    }

    string Packet::to_string()
    {
        string data;

        data += command_ + ';';

        for (auto &arg: args_) {
            data += string(arg->name) + '(' + string(rsvp_type_str[arg->type]) + ")=";

            switch (arg->type) {
                case INT:
                    data += std::to_string(arg->value.i);
                    break;
                case DBL:
                    data += std::to_string(arg->value.d);
                    break;
                case BOOL:
                    if (arg->value.b)
                        data += "TRUE";
                    else
                        data += "FALSE";
                    break;
                case VECD:
                    data += "[";

                    for (int i = 0; i < arg->len; i++) {
                        rsvp_vecd_elem_t *vecd_elem = rsvp_var_get_vecd_elem(arg, i);
                        data += string(vecd_elem->name) + "=" + std::to_string(vecd_elem->value) + ",";
                    }

                    data.replace(data.length() - 1, 1, "]");
                    break;
                case VECS:

                    for (int i = 0; i < arg->len; i++) {
                        data += string(arg->value.vecs[i]) + '\t';
                    }
                    break;
                default:
                    data += arg->value.s;
                    break;
            }

            data += ";";
        }

        data.erase(data.length() - 1);

        return data;
    }

    Packet Packet::parse_data(std::string data)
    {
        rsvp_data_t *raw_data = rsvp_data_parse((char*)data.c_str());
        string cmd(raw_data->cmd);
        Packet pkt(cmd);

        for (unsigned int i = 0; i < raw_data->args_count; i++) {
            rsvp_var_t *arg = raw_data->args[i];
            switch (arg->type)
            {
            case STR:
                pkt.add_arg(string(arg->name), string(arg->value.s));
                break;
            case INT:
                pkt.add_arg(arg->name, arg->value.i);
                break;
            case BOOL:
                pkt.add_arg(arg->name, arg->value.b);
                break;
            case DBL:
                pkt.add_arg(arg->name, arg->value.d);
                break;
            case VECD:
            {
                vector<tuple<string, double>> vecd;

                for (int i = 0; i < arg->len; i++) {
                    rsvp_vecd_elem_t* elem = rsvp_var_get_vecd_elem(arg, i);
                    vecd.push_back({elem->name, elem->value});
                }

                pkt.add_arg(arg->name, vecd);

                break;
            }
            case VECS:
            {
                vector<string> vecs;

                for (int i = 0; i < arg->len; i++) {
                    vecs.push_back(arg->value.vecs[i]);
                }

                pkt.add_arg(arg->name, vecs);
                break;
            }
            default:
                break;
            }
        }
        
        rsvp_data_free(raw_data);

        return pkt;
    }

    void Packet::add_arg(std::string name, int value) 
    {
        rsvp_var_t* var = rsvp_var_create_int((char*)name.c_str(), value);
        if (!var) {
            throw PacketException(rsvp_error_str());
        }
        args_.push_back(var);
    }

    void Packet::add_arg(std::string name, double value)
    {
        rsvp_var_t* var = rsvp_var_create_double((char*)name.c_str(), value);
        if (!var) {
            throw PacketException(rsvp_error_str());
        }
        args_.push_back(var);
    }

    void Packet::add_arg(std::string name, bool value)
    {
        rsvp_var_t* var = rsvp_var_create_bool((char*)name.c_str(), value);
        if (!var) {
            throw PacketException(rsvp_error_str());
        }
        args_.push_back(var);
    }

    void Packet::add_arg(std::string name, std::string value)
    {
        rsvp_var_t* var = rsvp_var_create_string((char*)name.c_str(), (char*)value.c_str());
        if (!var) {
            throw PacketException(rsvp_error_str());
        }
        args_.push_back(var);
    }

    void Packet::add_arg(std::string name, std::vector<std::tuple<string, double>> values)
    {
        rsvp_var_t* var = rsvp_var_create_vecd((char*)name.c_str());
        if (!var) {
            throw PacketException(rsvp_error_str());
        }

        for (auto &value: values) {
            if (rsvp_var_vecd_add_elem(var, (char*)get<0>(value).c_str(), get<1>(value)) < 0) {
                throw PacketException(rsvp_error_str());
            }
        }
        args_.push_back(var);
    }

    void Packet::add_arg(std::string name, std::vector<std::string> values)
    {
        rsvp_var_t* var = rsvp_var_create_vecs((char*)name.c_str());
         if (!var) {
            return;
        }

        for (auto &value: values) {
            if (rsvp_var_vecs_add_elem(var, (char*)value.c_str()) < 0) {
                break;
            }
        }
        args_.push_back(var);
    }
}