#pragma once

#include <vector>
#include <string>
#include <stdexcept>
#include <tuple>

extern "C" {
#include "rsvp.h"
}

namespace RSVP {

class PacketException : public std::exception {
    private:
    std::string msg_;

    public:
    PacketException(std::string msg) : msg_(msg) {}
    virtual const char* what() const throw() {
        return msg_.c_str();
    }
};

/**
 * @brief RSVP packet class
 * 
 */
class Packet {

    private:

    std::string command_;
    std::vector<rsvp_var_t*> args_;

    public:

    /**
     * @brief Construct a new RSVP Packet object
     * 
     * @param command Command for this RSVP packet
     */
    Packet(std::string command);

    /**
     * @brief Destroy the Packet object
     * 
     */
    ~Packet();

    /**
     * @brief Get the RSVP packet in string format
     * 
     * @return std::string Serialized packet ready to be sent
     */
    std::string to_string();
    
    /**
     * @brief Parse a RSVP packet
     * 
     * @param data RSVP string containing the serialized RSVP data
     * @return Packet 
     */
    static Packet parse_data(std::string data);

    /**
     * @brief Add an integer variable to the RSVP packet
     * 
     * @param name Variable name
     * @param value Integer value
     */
    void add_arg(std::string name, int value);

    /**
     * @brief Add a double variable to the RSVP packet
     * 
     * @param name Variable name
     * @param value Double value
     */
    void add_arg(std::string name, double value);

    /**
     * @brief Add a boolean variable to the RSVP packet
     * 
     * @param name Variable name
     * @param value Boolean value
     */
    void add_arg(std::string name, bool value);

    /**
     * @brief Add a string variable to the RSVP packet
     * 
     * @param name Variable name
     * @param value String value
     */
    void add_arg(std::string name, std::string value);

    /**
     * @brief Add a array of doubles variable to the RSVP packet
     * 
     * @param name Variable name
     * @param values A array of name value pairs
     */
    void add_arg(std::string name, std::vector<std::tuple<std::string, double>> values);

    /**
     * @brief Add a array of strings variable to the RSVP packet
     * 
     * @param name Variable name
     * @param values Array of strings
     */
    void add_arg(std::string name, std::vector<std::string> values);
    
};

} //namespace RSVP

