from PySide6.QtCore import Qt
import serial.tools
import serial.tools.list_ports
from frontend.ui.robot import Ui_MainWindow
from pyspj.spj import SphericalParallelJoint, WorldPosition

from PySide6.QtWidgets import (
    QApplication,  QMainWindow, QMessageBox, QPushButton, QSpinBox, QComboBox
)

from PySide6.QtGui import QCloseEvent, QImage, QPixmap
from PySide6.QtCore import QObject, Qt, QThread, Signal as pyqtSignal
import time, sys
from functools import partial
from enum import Enum
import serial
import platform
from pathlib import Path

POS_DEFAULT = 0.0
VEL_DEFAULT = 12.0
ACC_DEFAULT = 18.0

class JOG(Enum):
    RZ_NEG = 0
    RZ_POS = 1
    RY_NEG = 2
    RY_POS = 3
    RX_NEG = 4
    RX_POS = 5

class RobotThread(QThread):

    update_positions = pyqtSignal(list)
    _stop = False

    def __init__(self, spj: SphericalParallelJoint, parent = None) -> None:
        super().__init__(parent)
        self._spj = spj

    def send_pos(self, move : WorldPosition):
        self._spj.move_async(move)

    def stop(self):
        self._stop = True

    def run(self):

        while not self._stop:
            # Get current position
            pos = self._spj.get_current_position()
            self.update_positions.emit(pos)
        
class Window(QMainWindow, Ui_MainWindow):

    def __init__(self, parent = None) -> None:
        
        super().__init__(parent)
        
        self.setupUi(self)
        
        self._com_ports: list[tuple[str, str]] = []
        self.__setup_comports()
        
        self.__setup_signals()

        self.step_size_cb.addItems(["1", "5", "10", "50"])
        
        self._spj: SphericalParallelJoint = SphericalParallelJoint()
        self._angle_lut : list[list] = None
        self._step_lut: list[list] = None
        
        # self._angle_lut, self._step_lut= self._spj.load_lookup_table(Path("../pyspj/spj_lut.json"))
        
        self.current_pos = None
        
        self.__enable_robot_controls(False)
        
        self._com_thread: RobotThread = None
        
    def __setup_comports(self):
        
        self.comports_comboBox.clear()
        
        coms = sorted(serial.tools.list_ports.comports())
        
        for com in coms:
            if com.hwid != "n/a":
                match platform.system():
                    case "Linux":
                        self._com_ports.append((f"/dev/{com.name}", com.description))
                    case "Windows":
                        self._com_ports.append((com.name, com.description))
                        
        self.comports_comboBox.addItems([x[0] for x in self._com_ports])
        self.comport_description_lineEdit.setText(self._com_ports[0][1])
    
    def __setup_signals(self):

        # UI
        self.move_pb.pressed.connect(self.move_pb_callback)
        self.home_pb.pressed.connect(self.home)
        self.store_pb.pressed.connect(self.store)
        self.step_size_cb.currentIndexChanged.connect(self.__update_spinbox_step_size)
        self.jog_rz_neg_pb.pressed.connect(partial(self.jog_callback, JOG.RZ_NEG))
        self.jog_rz_pos_pb.pressed.connect(partial(self.jog_callback, JOG.RZ_POS))
        self.jog_ry_neg_pb.pressed.connect(partial(self.jog_callback, JOG.RY_NEG))
        self.jog_ry_pos_pb.pressed.connect(partial(self.jog_callback, JOG.RY_POS))
        self.jog_rx_neg_pb.pressed.connect(partial(self.jog_callback, JOG.RX_NEG))
        self.jog_rx_pos_pb.pressed.connect(partial(self.jog_callback, JOG.RX_POS))

        self.comports_comboBox.currentIndexChanged.connect(self.__update_comport_description)
        self.comport_connect_pushButton.clicked.connect(self.__connect_robot)
        
        # Thread
        # self.robot.update_positions.connect(self.update_position)

    def __update_spinbox_step_size(self):

        self.move_rx_sb.setSingleStep(int(self.step_size_cb.currentText()))
        self.move_ry_sb.setSingleStep(int(self.step_size_cb.currentText()))
        self.move_rz_sb.setSingleStep(int(self.step_size_cb.currentText()))

    def __update_comport_description(self):
        
        self.comport_description_lineEdit.setText(self._com_ports[self.comports_comboBox.currentIndex()][1])
    
    def __enable_robot_controls(self, enable: bool = True):
        
        for button in self.robot_control_gridLayout.parentWidget().findChildren(QPushButton):
            _button: QPushButton = button
            _button.setEnabled(enable)
            
        for comboBox in self.robot_control_gridLayout.parentWidget().findChildren(QComboBox):
            _comboBox: QComboBox = comboBox
            _comboBox.setEnabled(enable)
            
        for spinBox in self.robot_control_gridLayout.parentWidget().findChildren(QSpinBox):
            _spinbox: QSpinBox = spinBox
            _spinbox.setEnabled(enable)
    
    def __connect_robot(self):
        
        match self.comport_connect_pushButton.text():
            case "Connect":
                
                try:
                    self._spj.connect(self._com_ports[self.comports_comboBox.currentIndex()][0])
                except RuntimeError as ex:
                    self.err_msg(ex.__str__())
                    return
                
                # Unlock all robot controls
                self.__enable_robot_controls(True)
                self.comport_connect_pushButton.setText("Disconnect")
                self._com_thread = RobotThread(self._spj)
                self._com_thread.update_positions.connect(self.update_position)
                self._com_thread.start()
                self.home()
            
            case "Disconnect":
                
                self.store()
                self._com_thread.stop()
                self._com_thread.wait()
                self._spj.close()
                self.__enable_robot_controls(False)
                self.comport_connect_pushButton.setText("Connect")
    
    def home(self):
        self.move(SphericalParallelJoint.home_pos())

    def store(self):
        self.move(SphericalParallelJoint.store_pos())

    def jog_callback(self, jog: JOG):

        new_pos = self.current_pos
        step = int(self.step_size_cb.currentText())
        
        match jog:
            case JOG.RZ_NEG:
                new_pos.rz -= step
            case JOG.RZ_POS:
                new_pos.rz += step
            case JOG.RY_NEG:
                new_pos.ry -= step
            case JOG.RY_POS:
                new_pos.ry += step
            case JOG.RX_NEG:
                new_pos.rx -= step
            case JOG.RX_POS:
                new_pos.rx += step

        self.move(new_pos)

    def move_pb_callback(self):

        move = WorldPosition(
            float(self.move_rx_sb.value()),
            float(self.move_ry_sb.value()),
            float(self.move_rz_sb.value())
        )
        self.move(move)

    def move(self, new_pos: WorldPosition):

        self.current_pos = new_pos
        self.update_spinboxes(new_pos)
        self._com_thread.send_pos(new_pos)

        # print(self.current_pos)

    def update_spinboxes(self, pos : WorldPosition):

        self.move_rx_sb.setValue(int(pos.rx))
        self.move_ry_sb.setValue(int(pos.ry))
        self.move_rz_sb.setValue(int(pos.rz))

    def update_position(self, pos : list[float]):

        # pos = list(map(int, pos))
        # index = self._step_lut.index(pos)
        # angle = self._angle_lut[index]
        # 
        try:
            self.current_rx_le.setText(str(int(pos[2])))
            self.current_ry_le.setText(str(int(pos[1])))
            self.current_rz_le.setText(str(int(pos[0])))
        except ValueError as ex:
            # print(ex)
            return

    def closeEvent(self, a0: QCloseEvent) -> None:

        self.store()
        self._com_thread.stop()
        self._com_thread.wait()
        self._spj.close()

        return super().closeEvent(a0)

    def err_msg(self, msg: str):
        
        QMessageBox.critical(self, "Error", msg)
    
if __name__ == "__main__":
    
    import os
    
    os.chdir(os.path.dirname(__file__))
    
    app = QApplication(sys.argv)
    win = Window()
    win.show()
    sys.exit(app.exec())
