from PySide6.QtCore import Qt
from frontend.ui.robot_ui import Ui_MainWindow
from pyspj.spj import SphericalParallelJoint, WorldPosition

from PySide6.QtWidgets import (
    QApplication,  QMainWindow, QMessageBox, QFileDialog, QWidget
)

from PySide6.QtGui import QCloseEvent, QImage, QPixmap
from PySide6.QtCore import QObject, Qt, QThread, Signal as pyqtSignal
import time, sys
from functools import partial
from enum import Enum

POS_DEFAULT = 0.0
VEL_DEFAULT = 12.0
ACC_DEFAULT = 18.0

class JOG(Enum):
    RZ_NEG = 0
    RZ_POS = 1
    RY_NEG = 2
    RY_POS = 3
    RZ1_NEG = 4
    RZ1_POS = 5

class CommThread(QThread):

    update_positions = pyqtSignal(list)
    _stop = False

    def __init__(self, serial : str, parent = None) -> None:
        super().__init__(parent)
        self._spj = SphericalParallelJoint(serial)
        # values = self._spj.get_current_position()

    def send_pos(self, move : WorldPosition):
        self._spj.move(move)

    def stop(self):
        self._stop = True

    def run(self):

        while not self._stop:
            # Get current position
            pos = self._spj.get_current_position()
            self.update_positions.emit(pos)

class Window(QMainWindow, Ui_MainWindow):

    def __init__(self, parent = None) -> None:
        super().__init__(parent)
        self.setupUi(self)
        self.robot = CommThread("COM9")
        self.__setup_signals()
        self.robot.start()
        time.sleep(0.5)
        self.step_size_cb.addItems(["1", "5", "10", "50"])
        self.current_pos = None
        self.store()

    def __setup_signals(self):

        # UI
        self.move_pb.pressed.connect(self.move_pb_callback)
        self.home_pb.pressed.connect(self.home)
        self.store_pb.pressed.connect(self.store)
        self.step_size_cb.currentIndexChanged.connect(self.__update_spinbox_step_size)
        self.jog_rz_neg_pb.pressed.connect(partial(self.jog_callback, JOG.RZ_NEG))
        self.jog_rz_pos_pb.pressed.connect(partial(self.jog_callback, JOG.RZ_POS))
        self.jog_ry_neg_pb.pressed.connect(partial(self.jog_callback, JOG.RY_NEG))
        self.jog_ry_pos_pb.pressed.connect(partial(self.jog_callback, JOG.RY_POS))
        self.jog_rz1_neg_pb.pressed.connect(partial(self.jog_callback, JOG.RZ1_NEG))
        self.jog_rz1_pos_pb.pressed.connect(partial(self.jog_callback, JOG.RZ1_POS))

        # Thread
        self.robot.update_positions.connect(self.update_position)

    def __update_spinbox_step_size(self):

        self.move_rx_sb.setSingleStep(int(self.step_size_cb.currentText()))
        self.move_ry_sb.setSingleStep(int(self.step_size_cb.currentText()))
        self.move_rz_sb.setSingleStep(int(self.step_size_cb.currentText()))

    def home(self):
        self.move(WorldPosition(0.0, 0.0, 0.0))

    def store(self):
        self.move(WorldPosition(-60.0, 0.0, 0.0))

    def jog_callback(self, jog: JOG):

        new_pos = self.current_pos
        step = int(self.step_size_cb.currentText())
        match jog:
            case JOG.RZ_NEG:
                new_pos.rz -= step
            case JOG.RZ_POS:
                new_pos.rz += step
            case JOG.RY_NEG:
                new_pos.ry -= step
            case JOG.RY_POS:
                new_pos.ry += step
            case JOG.RZ1_NEG:
                new_pos.rz1 -= step
            case JOG.RZ1_POS:
                new_pos.rz1 += step

        self.move(new_pos)

    def move_pb_callback(self):

        move = WorldPosition(
            float(self.move_rx_sb.value()),
            float(self.move_ry_sb.value()),
            float(self.move_rz_sb.value())
        )
        self.move(move)

    def move(self, new_pos: WorldPosition):

        self.current_pos = new_pos
        self.update_spinboxes(new_pos)
        self.robot.send_pos(new_pos)

        print(self.current_pos)

    def update_spinboxes(self, pos : WorldPosition):

        self.move_rx_sb.setValue(int(pos.rz))
        self.move_ry_sb.setValue(int(pos.ry))
        self.move_rz_sb.setValue(int(pos.rz1))

    def update_position(self, pos : list[float]):

        try:
            self.current_rx_le.setText(str(int(pos[0])))
            self.current_ry_le.setText(str(int(pos[1])))
            self.current_rz_le.setText(str(int(pos[2])))
        except ValueError as ex:
            print(ex)

    def closeEvent(self, a0: QCloseEvent) -> None:

        self.store()
        self.robot.stop()

        # Wait for thread to stop
        while(self.robot.isRunning()):
            time.sleep(0.001)

        return super().closeEvent(a0)

if __name__ == "__main__":
    app = QApplication(sys.argv)
    win = Window()
    win.show()
    sys.exit(app.exec())
